<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * WC_Gateway_MomentPay Payment Gateway Class.
 *
 * @extends WC_Payment_Gateway
 */
class WC_Gateway_MomentPay extends WC_Payment_Gateway {

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		$this->id                 = 'momentpay';
		$this->icon               = null; // You can add an icon later
		$this->has_fields         = false; // We will use a redirect, so no fields on checkout page
		$this->method_title       = 'MomentPay'; // Title shown in WooCommerce admin
		$this->method_description = 'Accept payments via MomentPay payment gateway.'; // Description shown in WooCommerce admin

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		// Define user set variables
		$this->title        = $this->get_option( 'title' );
		$this->description  = $this->get_option( 'description' );
		$this->enabled      = $this->get_option( 'enabled' );
		$this->merchant_id  = $this->get_option( 'merchant_id' );
		$this->credentials_user = $this->get_option( 'credentials_user' );
		$this->credentials_key = $this->get_option( 'credentials_key' );
		$this->version      = $this->get_option( 'version' );
		$this->action_url   = $this->get_option( 'action_url' );
		// Add more settings as needed based on your JavaScript 'params'

		// Actions
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		// Add action to handle payment responses (we'll need to figure out the best approach for this based on MomentPay's response mechanism)
		// add_action( 'woocommerce_api_wc_gateway_momentpay', array( $this, 'handle_payment_response' ) ); // Example - adjust if needed
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields() {
		$this->form_fields = array(
			'enabled' => array(
				'title'   => __( 'Enable/Disable', 'woocommerce-momentpay' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable MomentPay Payment Gateway', 'woocommerce-momentpay' ),
				'default' => 'yes',
				'description' => __('Enable to use MomentPay payment gateway', 'woocommerce-momentpay'),
			),
			'title' => array(
				'title'       => __( 'Title', 'woocommerce-momentpay' ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', 'woocommerce-momentpay' ),
				'default'     => __( 'MomentPay', 'woocommerce-momentpay' ),
				'desc_tip'    => true,
			),
			'description' => array(
				'title'       => __( 'Description', 'woocommerce-momentpay' ),
				'type'        => 'textarea',
				'description' => __( 'Payment method description that the customer will see on your checkout.', 'woocommerce-momentpay' ),
				'default'     => __( 'Pay securely via MomentPay.', 'woocommerce-momentpay' ),
				'desc_tip'    => true,
			),
			'merchant_id' => array(
				'title'       => __( 'Merchant ID/Username', 'woocommerce-momentpay' ),
				'type'        => 'text',
				'description' => __( 'Enter your MomentPay Merchant ID or Username.', 'woocommerce-momentpay' ),
				'desc_tip'    => true,
				'required'    => true,
			),
			'credentials_user' => array(
				'title'       => __( 'Credentials User', 'woocommerce-momentpay' ),
				'type'        => 'text',
				'description' => __( 'Enter your MomentPay Credentials User (for HISV2).', 'woocommerce-momentpay' ),
				'desc_tip'    => true,
				'required'    => false, // Make required if you are primarily using HISV2
			),
			'credentials_key' => array(
				'title'       => __( 'Credentials Key', 'woocommerce-momentpay' ),
				'type'        => 'text',
				'description' => __( 'Enter your MomentPay Credentials Key (for HISV2).', 'woocommerce-momentpay' ),
				'desc_tip'    => true,
				'required'    => false, // Make required if you are primarily using HISV2
			),
			'version' => array(
				'title'       => __( 'Version', 'woocommerce-momentpay' ),
				'type'        => 'select',
				'description' => __( 'Select the MomentPay API version you are using.', 'woocommerce-momentpay' ),
				'default'     => 'default',
				'options'     => array(
					'default' => __('Default', 'woocommerce-momentpay'), // or specify your default version name
					'HISV2'   => __('HISV2', 'woocommerce-momentpay'),
				),
				'desc_tip'    => true,
			),
			'action_url' => array(
				'title'       => __( 'Action URL', 'woocommerce-momentpay' ),
				'type'        => 'text',
				'description' => __( 'Enter the Action URL provided by MomentPay (e.g., for testing or live).', 'woocommerce-momentpay' ),
				'default'     => 'https://testapp.ariticapp.com/ma/provider/iframe', // Use your default test URL
				'desc_tip'    => true,
				'required'    => true,
			),
			// Add more settings fields here if needed, based on your JS 'params'
		);
	}

	/**
	 * Output payment fields for the frontend.
	 */
	public function payment_fields() {
		if ( $this->description ) {
			// Display the description set in the admin, if any
			echo wpautop( wp_kses_post( $this->description ) );
		}
		// We might not need any actual fields here as the form is in the iframe.
		// But you could add some informational text or an icon here if you want.
	}

	/**
	 * Process the payment and return the result.
	 *
	 * @param int $order_id Order ID.
	 * @return array
	 */
	public function process_payment( $order_id ) {
		$order = wc_get_order( $order_id );

		// 1. Gather Data for Payload (from order and settings)
		$payload_data = $this->generate_payment_payload( $order );

		// 2. Generate Tab ID
		$tab_id = $this->generate_tab_id();

		// 3.  Output HTML & JavaScript to create iframe and submit form
		$this->generate_iframe_form( $order, $payload_data, $tab_id );

		// 4. Return a result indicating redirect (even though it's not a direct redirect in the traditional sense)
		return array(
			'result'   => 'success',
			'redirect' => '', // We are not doing a traditional redirect, the iframe form is submitted in place.
		);
	}


	/**
	 * Generate Payment Payload array based on settings and order.
	 *
	 * @param WC_Order $order WooCommerce Order object
	 * @return array Payload data array
	 */
	public function generate_payment_payload( $order ) {
		$version = $this->version;
		$payload = array();

		if ( $version === 'HISV2' ) {
			$payload = array(
				'credentials' => array(
					'user'    => $this->credentials_user,
					'key'     => $this->credentials_key,
					'version' => $version,
				),
				'username'        => $this->merchant_id,
				'customer_details' => array(
					array(
						'customer_name'  => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(), // or get_formatted_billing_full_name()
						'customer_id'    => $order->get_user_id(), // Or order number? Decide what to use as customer ID
						'payment_amount' => $order->get_total(),
						'customer_email'  => $order->get_billing_email(),
						'customer_phone'  => $order->get_billing_phone(),
					),
				),
				'processing_id'      => $order->get_order_number(), // Or generate a unique processing ID if needed
				'paymode'            => 'momentpay-pg', // Hardcoded as per your JS example
				'payment_fill'       => 'pre_full',      // Hardcoded as per your JS example
				'transaction_location' => '', //  From JS, needs clarification if this is dynamic
				'callback_url'         => '', // We will handle response via postMessage, so might not need these
				'redirect_url'         => '', // We will handle response via postMessage, so might not need these
				'location_id'          => '1', // From JS, needs clarification if this is dynamic
			);
		} else {
			// Default Payload structure (non-HISV2) - Adapt based on your gateway's API
			$payload = array(
				'auth' => array(
					'user' => '', // You might need to add 'authuser' and 'authkey' settings if used
					'key'  => '', // You might need to add 'authuser' and 'authkey' settings if used
				),
				'cashier_id'  => $this->merchant_id,
				'accounts'    => array(
					array(
						'patient_name'   => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
						'account_number' => $order->get_user_id(), // Or order number? Decide what to use as account_number
						'amount'         => $order->get_total(),
						'email'          => $order->get_billing_email(),
						'phone'          => $order->get_billing_phone(),
					),
				),
				'processing_id'    => $order->get_order_number(), // Or generate unique ID
				'paymode'          => 'momentpay-pg', // Hardcoded
				'payment_fill'     => 'pre_full',      // Hardcoded
				'payment_location' => '', // From JS, needs clarification if this is dynamic
				'return_url'       => '', // We will handle response via postMessage, might not need these
				'response_url'       => '', // We will handle response via postMessage, might not need these
				'tabId'            => '', // Will be added later
			);
		}

		//  You might need to add 'mid', 'check_sum_hash' parameters to the payload if your gateway requires them
		//  based on how they are used in your JS. We need more info on these.

		return $payload;
	}


	/**
	 * Generate a unique Tab ID.
	 *
	 * @return string
	 */
	public function generate_tab_id() {
		return time(); // Simple timestamp for now, can be improved for better uniqueness
	}


	/**
	 * Output the iframe form and related JavaScript to the checkout page.
	 *
	 * @param WC_Order $order        WooCommerce order object.
	 * @param array    $payload_data Generated payload data.
	 * @param string   $tab_id       Generated Tab ID.
	 */
	public function generate_iframe_form( $order, $payload_data, $tab_id ) {
		$action_url = $this->action_url; // Get action URL from settings
		$token_json = wp_json_encode( $payload_data ); // Encode payload to JSON

		// Hidden form fields to capture response (as per your JS example)
		?>
		<input type="hidden" id="amountField" name="mmntpay_amount" value="">
		<input type="hidden" id="mmntpay_paymentid" name="mmntpay_paymentid" value="">
		<input type="hidden" id="mmntpay_signature" name="mmntpay_signature" value="">
		<input type="hidden" id="mmntpay_response_status" name="mmntpay_response_status" value="">
		<button type="button" id="momentpay-paymentresponse" style="display:none;"></button> <?php // Hidden button

		?>
		<div id="momentpay-overlay" style="position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0, 0, 0, 0.7); z-index: 10000; display: flex; justify-content: center; alignItems: center; overflow: auto;">
			<button id="momentpay-close-button" style="position: absolute; top: 20px; right: 20px; z-index: 10001; cursor: pointer; border: 2px solid red; border-radius: 50%; width: 30px; height: 30px; line-height: 24px; text-align: center; background-color: white; color: red; font-size: 20px; padding: 0; display: flex; justify-content: center; align-items: center; outline: none;">&#x2715;</button>
			<iframe id="momentpay-iframe" style="width: 60%; height: 80%; border: none; background-color: white;"></iframe>
		</div>

		<script type="text/javascript">
			document.addEventListener('DOMContentLoaded', function() {
				var overlay = document.getElementById('momentpay-overlay');
				var closeButton = document.getElementById('momentpay-close-button');
				var iframe = document.getElementById('momentpay-iframe');

				// Close Iframe Function
				function closeIframe() {
					if (overlay) {
						overlay.remove();
						// Trigger onClose callback logic (if needed, you can refine this)
						document.getElementById('mmntpay_response_status').value = "1404";
						document.getElementById("momentpay-paymentresponse").click();
						console.log("Payment window closed");
					}
				}

				closeButton.onclick = closeIframe;


				var doc = iframe.contentDocument || iframe.contentWindow.document;
				doc.open();
				doc.close();

				var form = doc.createElement('form');
				form.method = 'post';
				form.action = '<?php echo esc_url( $action_url ); ?>';
				doc.body.appendChild(form);

				// Add hidden input for token (payload)
				var tokenInput = doc.createElement('input');
				tokenInput.type = 'hidden';
				tokenInput.name = 'token';
				tokenInput.value = '<?php echo esc_attr( $token_json ); ?>';
				form.appendChild(tokenInput);

				// Add other hidden inputs as needed (mid, check_sum_hash, tabId if applicable -  we need to clarify if these are needed and how to get them)
				// Example - if you have 'mid' and 'check_sum_hash' in your settings:
				<?php
					// Example - if you add 'mid' and 'checksum_hash' settings in init_form_fields() and save them as options:
					// $mid = $this->get_option('mid');
					// $checksum_hash = $this->get_option('checksum_hash');
					// if ($mid) :
					// 	echo "var midInput = doc.createElement('input');";
					// 	echo "midInput.type = 'hidden';";
					// 	echo "midInput.name = 'mid';";
					// 	echo "midInput.value = '" . esc_attr( $mid ) . "';";
					// 	echo "form.appendChild(midInput);";
					// endif;
					// if ($checksum_hash) :
					// 	echo "var checksumInput = doc.createElement('input');";
					// 	echo "checksumInput.type = 'hidden';";
					// 	echo "checksumInput.name = 'check_sum_hash';";
					// 	echo "checksumInput.value = '" . esc_attr( $checksum_hash ) . "';";
					// 	echo "form.appendChild(checksumInput);";
					// endif;
				?>
				var tabIdInput = doc.createElement('input');
				tabIdInput.type = 'hidden';
				tabIdInput.name = 'tabId';
				tabIdInput.value = '<?php echo esc_attr( $tab_id ); ?>';
				form.appendChild(tabIdInput);


				form.submit(); // Auto submit the iframe form

				// --- Response Handling ---
				window.addEventListener('message', function(event) {
					if (event.origin === '<?php echo parse_url(esc_url( $action_url ), PHP_URL_HOST) . "://" . parse_url(esc_url( $action_url ), PHP_URL_HOST); ?>') { // **Important:** Validate origin! Replace with your actual MomentPay domain
						if (typeof event.data === 'string') {
							console.log('Received data from iframe:', event.data);
							try {
								var responseData = JSON.parse(event.data);
								if (responseData.status_code === 1200) {
									// Handle successful payment
									console.log('Payment successful:', responseData);
									document.getElementById('amountField').value = responseData.payment_response.baseAmount; //Example, adapt to your response structure
									document.getElementById('mmntpay_paymentid').value = "ch_948749387493"; //  **Replace with actual payment ID from responseData**
									document.getElementById('mmntpay_signature').value = "ch_948749387493"; // **Replace with actual signature/processing ID from responseData if available**
									document.getElementById('mmntpay_response_status').value = responseData.status_code;
									document.getElementById("momentpay-paymentresponse").click(); // Trigger form submission to finalize order
								} else {
									document.getElementById('mmntpay_response_status').value = responseData.status_code;
									document.getElementById("momentpay-paymentresponse").click(); // Trigger form submission to finalize order (even on failure)
									alert('Your payment has failed. Please try again! Status Code: ' + responseData.status_code);
								}
							} catch (e) {
								console.error('Error parsing JSON response from iframe:', e);
								alert('Error processing payment response. Please contact support.');
							} finally {
								closeIframe(); // Close iframe after processing response (success or failure)
							}
						}
					}
				}, false);

			});
		</script>
		<?php
	}


	/**
	 * Handle frontend submission and process payment
	 * This function is triggered when the "Place order" button is clicked.
	 *
	 * @return void
	 */
	public function process_payment_submission() {
		// This function might be needed if you are using a standard WooCommerce form submission for final order placement
		// after the MomentPay iframe interaction.
		// You would handle capturing the hidden field values (mmntpay_paymentid, etc.) here and validate the payment.

		// For now, we might be able to rely on the JavaScript and the "momentpay-paymentresponse" button click
		// to handle the final form submission.  We need to understand the complete flow to finalize this.

		// Example:
		// if ( isset( $_POST['mmntpay_response_status'] ) && $_POST['mmntpay_response_status'] == '1200' ) {
		// 	// Payment was successful, you might want to verify signature etc. here before completing order.
		// 	return true; // Payment processed successfully
		// } else {
		// 	// Payment failed or was cancelled.
		// 	return false; // Indicate payment failure
		// }

		return true; // For now, assume success for testing plugin structure.
	}


	/**
	 * Can the order be refunded
	 * @param WC_Order $order Order object.
	 * @return bool
	 */
	public function can_refund_order( $order ) {
		return false; // For now, disable refunds via plugin. You can implement refund logic later if MomentPay API supports it.
	}


	/**
	 * Process refunds.
	 * If refunds are enabled via MomentPay API, implement the refund logic here.
	 * @param int    $order_id Order ID.
	 * @param float  $amount Refund amount.
	 * @param string $reason Refund reason.
	 * @return bool|WP_Error
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		return new WP_Error( 'momentpay_refunds_not_supported', __( 'MomentPay refunds are not supported by this plugin.', 'woocommerce-momentpay' ) );
	}


}